#!/usr/bin/env python3
import argparse
import json
import os
import requests
from datetime import datetime
from validate import validate_json

BASE_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
REQUESTS_DIR = os.path.join(BASE_DIR, "requests")
RESPONSES_DIR = os.path.join(BASE_DIR, "responses")
SCHEMAS_DIR = os.path.join(BASE_DIR, "schemas")

NODE_URL_LOCAL = "http://127.0.0.1:12345/connect"
NODE_URL_RPC = "http://rpc.cellframe.net"

def load_request_template(request_name):
    path = os.path.join(REQUESTS_DIR, f"{request_name}.json")
    with open(path, "r") as f:
        return json.load(f)

def apply_overrides(request_data, subcommand, args_list, add_version):
    if subcommand:
        request_data["subcommand"] = [subcommand]
    if args_list:
        request_data.setdefault("arguments", {})
        for key, value in args_list:
            request_data["arguments"][key] = value
    if add_version:
        request_data["version"] = 2
    return request_data

def send_request(data, use_rpc):
    url = NODE_URL_RPC if use_rpc else NODE_URL_LOCAL
    response = requests.post(url, json=data, timeout=10)
    response.raise_for_status()
    return response.json()

def save_response(response_data, name):
    os.makedirs(RESPONSES_DIR, exist_ok=True)
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    filename = f"{name}_{timestamp}.json"
    path = os.path.join(RESPONSES_DIR, filename)
    with open(path, "w") as f:
        json.dump(response_data, f, indent=2)
    print(f"✅ Saved to: {path}")
    return path

def main():
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(dest="command")

    send = subparsers.add_parser("send")
    send.add_argument("--request", required=True)
    send.add_argument("--subcommand")
    send.add_argument("--arg", action="append", nargs=2, metavar=("KEY", "VALUE"))
    send.add_argument("--v2", action="store_true")
    send.add_argument("--target", choices=["local", "rpc"], default="local")
    send.add_argument("--save", help="Override filename")
    send.add_argument("--validate", action="store_true")
    send.add_argument("--schema", help="Schema filename")
    send.add_argument("--schema-version", help="v1 or v2")

    validate = subparsers.add_parser("validate")
    validate.add_argument("--response", required=True)
    validate.add_argument("--schema", required=True)
    validate.add_argument("--schema-version", required=True)

    args = parser.parse_args()

    if args.command == "send":
        try:
            request = load_request_template(args.request)
            request = apply_overrides(request, args.subcommand, args.arg, args.v2)
            response = send_request(request, use_rpc=(args.target == "rpc"))
            print(json.dumps(response, indent=2))
            filename = args.save or args.request
            saved_path = save_response(response, filename)

            if args.validate:
                schema_path = os.path.join(SCHEMAS_DIR, args.schema_version, args.schema)
                valid = validate_json(saved_path, schema_path)
                if not valid:
                    exit(1)

        except Exception as e:
            print(f"❌ Error: {e}")
            exit(1)

    elif args.command == "validate":
        response_path = os.path.join(RESPONSES_DIR, args.response)
        schema_path = os.path.join(SCHEMAS_DIR, args.schema_version, args.schema)
        valid = validate_json(response_path, schema_path)
        if not valid:
            exit(1)
    else:
        parser.print_help()

if __name__ == "__main__":
    main()