#!/usr/bin/env python3
"""
Automated Cellframe Node Testing Script
Validates node responses against JSON schemas for backward compatibility
"""

import os
import sys
import json
import time
import argparse
import subprocess
from datetime import datetime
from pathlib import Path

# Add main directory to path
sys.path.insert(0, os.path.join(os.path.dirname(__file__), 'main'))
from client import send_request, load_request_template
from validate import validate_json

class NodeTester:
    def __init__(self, node_url="http://127.0.0.1:12345/connect", output_dir="test_results"):
        self.node_url = node_url
        self.output_dir = Path(output_dir)
        self.output_dir.mkdir(exist_ok=True)
        self.results = {"v1": {}, "v2": {}, "summary": {}}
        self.timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        
    def test_command(self, command, version="v1", args=None):
        """Test a single command with specified version"""
        print(f"🔄 Testing {command} (API {version})")
        
        try:
            # Load request template
            request_data = load_request_template(command)
            
            # Apply version and arguments
            if version == "v2":
                request_data["version"] = 2
            
            if args:
                request_data.setdefault("arguments", {})
                for key, value in args.items():
                    request_data["arguments"][key] = value
            
            # Send request
            response = send_request(request_data, use_rpc=False)
            
            # Save response
            response_file = self.output_dir / f"{command}_{version}_{self.timestamp}.json"
            with open(response_file, 'w') as f:
                json.dump(response, f, indent=2)
            
            # Validate against schema
            schema_path = f"schemas/{version}/{command}.schema.json"
            if os.path.exists(schema_path):
                validation_result = validate_json(str(response_file), schema_path, version)
                self.results[version][command] = {
                    "status": "PASS" if validation_result else "FAIL",
                    "response_file": str(response_file),
                    "schema": schema_path,
                    "validation": validation_result
                }
                print(f"✅ {command} ({version}): PASS" if validation_result else f"❌ {command} ({version}): FAIL")
            else:
                self.results[version][command] = {
                    "status": "NO_SCHEMA",
                    "response_file": str(response_file),
                    "schema": None,
                    "validation": None
                }
                print(f"⚠️ {command} ({version}): NO SCHEMA")
                
        except Exception as e:
            self.results[version][command] = {
                "status": "ERROR",
                "error": str(e),
                "response_file": None,
                "schema": schema_path if 'schema_path' in locals() else None,
                "validation": False
            }
            print(f"💥 {command} ({version}): ERROR - {e}")
    
    def run_all_tests(self):
        """Run all available tests for both API versions"""
        print("🚀 Starting Cellframe Node API validation tests...")
        print(f"📍 Node URL: {self.node_url}")
        print(f"📂 Output directory: {self.output_dir}")
        print("=" * 50)
        
        # Get all available request templates
        requests_dir = Path("requests")
        commands = [f.stem for f in requests_dir.glob("*.json")]
        
        # Test commands that work without arguments first
        basic_commands = ["version", "net_list", "mempool_count", "node_stats"]
        
        for version in ["v1", "v2"]:
            print(f"\n📋 Testing API {version}")
            print("-" * 30)
            
            for command in basic_commands:
                if command in commands:
                    self.test_command(command, version)
                    time.sleep(0.5)  # Small delay between requests
        
        # Test wallet commands with default arguments
        wallet_commands = {
            "wallet_list": {},
            "wallet_info_w": {"w": "main"},
            "wallet_outputs_w": {"net": "KelVPN", "w": "main", "token": "KEL"}
        }
        
        for version in ["v1", "v2"]:
            for command, args in wallet_commands.items():
                if command.replace("_w", "") in commands or command in commands:
                    try:
                        self.test_command(command, version, args)
                    except:
                        print(f"⚠️ Skipping {command} - wallet may not exist")
                    time.sleep(0.5)
    
    def generate_report(self):
        """Generate detailed test report"""
        report_file = self.output_dir / f"test_report_{self.timestamp}.json"
        
        # Calculate summary statistics
        total_tests = 0
        passed_tests = 0
        failed_tests = 0
        error_tests = 0
        no_schema_tests = 0
        
        for version in ["v1", "v2"]:
            for command, result in self.results[version].items():
                total_tests += 1
                if result["status"] == "PASS":
                    passed_tests += 1
                elif result["status"] == "FAIL":
                    failed_tests += 1
                elif result["status"] == "ERROR":
                    error_tests += 1
                elif result["status"] == "NO_SCHEMA":
                    no_schema_tests += 1
        
        self.results["summary"] = {
            "timestamp": self.timestamp,
            "node_url": self.node_url,
            "total_tests": total_tests,
            "passed": passed_tests,
            "failed": failed_tests,
            "errors": error_tests,
            "no_schema": no_schema_tests,
            "success_rate": f"{(passed_tests/total_tests*100):.1f}%" if total_tests > 0 else "0%"
        }
        
        # Save detailed report
        with open(report_file, 'w') as f:
            json.dump(self.results, f, indent=2)
        
        # Print summary
        print("\n" + "=" * 50)
        print("📊 TEST RESULTS SUMMARY")
        print("=" * 50)
        print(f"🕒 Timestamp: {self.timestamp}")
        print(f"🌐 Node URL: {self.node_url}")
        print(f"📈 Total tests: {total_tests}")
        print(f"✅ Passed: {passed_tests}")
        print(f"❌ Failed: {failed_tests}")
        print(f"💥 Errors: {error_tests}")
        print(f"⚠️ No schema: {no_schema_tests}")
        print(f"📊 Success rate: {self.results['summary']['success_rate']}")
        print(f"📄 Detailed report: {report_file}")
        
        # Return exit code for CI/CD
        return 0 if failed_tests == 0 and error_tests == 0 else 1

def main():
    parser = argparse.ArgumentParser(description="Test Cellframe Node API responses")
    parser.add_argument("--node-url", default="http://127.0.0.1:12345/connect",
                       help="Node URL to test (default: local node)")
    parser.add_argument("--output-dir", default="test_results",
                       help="Output directory for test results")
    parser.add_argument("--command", help="Test specific command only")
    parser.add_argument("--version", choices=["v1", "v2"], help="Test specific API version only")
    
    args = parser.parse_args()
    
    tester = NodeTester(args.node_url, args.output_dir)
    
    if args.command:
        # Test specific command
        version = args.version or "v1"
        tester.test_command(args.command, version)
    else:
        # Run all tests
        tester.run_all_tests()
    
    # Generate report and exit with appropriate code
    exit_code = tester.generate_report()
    sys.exit(exit_code)

if __name__ == "__main__":
    main()
